/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */


#include "IAmCommandReceiverShadow.h"
#include <string>
#include <fstream>
#include <stdexcept>
#include <cassert>
//#include "audiomanagertypes.h"
#include "CAmCommandSenderDbus.h"
#include "CAmDltWrapper.h"

#include "commandplugin_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AMCOMMANDPLUGIN
#include "trcGenProj/Header/IAmCommandReceiverShadow.cpp.trc.h"
#endif

using namespace am;


/**
 * static ObjectPathTable is needed for DBus Callback handling
 */
static DBusObjectPathVTable gObjectPathVTable;

IAmCommandReceiverShadow::IAmCommandReceiverShadow() :
        mFunctionMap(createMap()), //
        mDBUSMessageHandler(), //
        mpIAmCommandReceive(NULL), //
        mpCAmDbusWrapper(NULL)
{
    ETG_TRACE_USR1(("IAmCommandReceiverShadow::CommandReceiverShadow constructor called"))
}

IAmCommandReceiverShadow::~IAmCommandReceiverShadow()
{
    ETG_TRACE_USR1(("IAmCommandReceiverShadow::CommandReceiverShadow destructed"))

    mpIAmCommandReceive = NULL;
    mpCAmDbusWrapper = NULL;
}

void IAmCommandReceiverShadow::connect(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sourceID_t sourceID = (am_sourceID_t) mDBUSMessageHandler.getUInt();
    am_sinkID_t sinkID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    am_mainConnectionID_t mainConnectionID = 0;

   ETG_TRACE_USR1(("IAmCommandReceiverShadow::connect called, sourceID:%u sinkID:%u",sourceID,sinkID));

   am_Error_e returnCode = E_NON_EXISTENT;
  if((sourceID) && (sinkID))
  {
    returnCode = mpIAmCommandReceive->connect(sourceID, sinkID, mainConnectionID);
  }
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append((dbus_uint16_t) mainConnectionID);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::disconnect(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_mainConnectionID_t mainConnnectionID = (am_mainConnectionID_t) mDBUSMessageHandler.getUInt();
    am_Error_e returnCode = E_NON_EXISTENT;

   ETG_TRACE_USR1(("IAmCommandReceiverShadow::disconnect called, mainConnnectionID:%u", mainConnnectionID));

  if(mainConnnectionID)
  {
    returnCode = mpIAmCommandReceive->disconnect(mainConnnectionID);
  }
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::setVolume(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sinkID_t sinkID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    am_VolumeChangeMode_e mode = (am_VolumeChangeMode_e) mDBUSMessageHandler.getUInt();
    int16_t value = mDBUSMessageHandler.getInt();

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::setVolume called, sinkID:%u mode:%u", sinkID, (uint16_t)mode))

  am_Error_e returnCode = E_NON_EXISTENT;
  if(sinkID)
  {
    returnCode = E_OUT_OF_RANGE;

    if(value >= 0)
    {
      switch(mode)
      {
      case VCM_INCREMENT:  returnCode = mpIAmCommandReceive->volumeStep(sinkID,value);  break;
      case VCM_DECREMENT: returnCode = mpIAmCommandReceive->volumeStep(sinkID,(int16_t)(value * -1)); break;
      case VCM_ABSOLUTE: returnCode = mpIAmCommandReceive->setVolume(sinkID, value);   break;
      default: break;
      }
    }
  }

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::volumeStep(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sinkID_t sinkID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    int16_t volStep = (int16_t) mDBUSMessageHandler.getInt();
  am_Error_e returnCode = E_NON_EXISTENT;

   ETG_TRACE_USR2(("IAmCommandReceiverShadow::volumeStep called, sinkID:%u volStep:%d", sinkID,volStep))

  if(sinkID)
  {
    returnCode = mpIAmCommandReceive->volumeStep(sinkID, volStep);
  }
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::setSinkMuteState(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sinkID_t sinkID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    am_MuteState_e muteState = (am_MuteState_e) mDBUSMessageHandler.getInt();
    am_Error_e returnCode = E_NON_EXISTENT;

   ETG_TRACE_USR2(("IAmCommandReceiverShadow::setSinkMuteState called, sinkID:%u muteState:%u", sinkID, (uint16_t)muteState));

  if(sinkID)
  {
    returnCode = mpIAmCommandReceive->setSinkMuteState(sinkID, muteState);
  }
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::setMainSinkSoundProperty(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sinkID_t sinkID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    dbus_int16_t type = 0;
    dbus_int16_t value = 0;
    mDBUSMessageHandler.getProperty(type, value);
    am_MainSoundProperty_s mainSoundProperty;
    mainSoundProperty.type = (am_CustomMainSoundPropertyType_t) type;
    mainSoundProperty.value = (int32_t) value;
    am_Error_e returnCode = E_NON_EXISTENT;

   ETG_TRACE_USR2(("IAmCommandReceiverShadow::setMainSinkSoundProperty called, sinkID:%u SoundProperty Type::%u Value:%d"
         ,sinkID,(uint16_t)mainSoundProperty.type,mainSoundProperty.value))

  if(sinkID)
  {
    returnCode = mpIAmCommandReceive->setMainSinkSoundProperty(mainSoundProperty, sinkID);
  }
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::setMainSourceSoundProperty(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sourceID_t sourceID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    dbus_int16_t type = 0;
    dbus_int16_t value = 0;
    mDBUSMessageHandler.getProperty(type, value);
    am_MainSoundProperty_s mainSoundProperty;
    mainSoundProperty.type = (am_CustomMainSoundPropertyType_t) type;
    mainSoundProperty.value = (int32_t) value;
    am_Error_e returnCode = E_NON_EXISTENT;

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::setMainSourceSoundProperty called: sourceID:%u SoundProperty Type::%u Value:%d"
          ,sourceID,(uint16_t)mainSoundProperty.type,mainSoundProperty.value))

  if(sourceID)
  {
    returnCode = mpIAmCommandReceive->setMainSourceSoundProperty(mainSoundProperty, sourceID);
  }
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::setSystemProperty(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    dbus_int16_t type = 0;
    dbus_int16_t value = 0;
    mDBUSMessageHandler.getProperty(type, value);
    am_SystemProperty_s systemProperty;
    systemProperty.type = (am_CustomSystemPropertyType_t) type;
    systemProperty.value = (int32_t) value;

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::setSystemProperty called, SystemProperty Type::%u Value:%d"
         , (uint16_t)systemProperty.type,systemProperty.value))

    am_Error_e returnCode = mpIAmCommandReceive->setSystemProperty(systemProperty);
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListMainConnections(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive_NoArgs(msg);

    std::vector<am_MainConnectionType_s> listMainConnections;
    am_Error_e returnCode = mpIAmCommandReceive->getListMainConnections(listMainConnections);

    ETG_TRACE_USR4(("IAmCommandReceiverShadow::getListMainConnections called, size is %u", listMainConnections.size()));

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listMainConnections);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListMainSinks(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive_NoArgs(msg);
    std::vector<am_SinkType_s> listSinks;
    am_Error_e returnCode = mpIAmCommandReceive->getListMainSinks(listSinks);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListMainSinks called, size is %u" , listSinks.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listSinks);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListMainSources(DBusConnection* /* conn */, DBusMessage *msg)
{

    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    std::vector<am_SourceType_s> listSources;
    mDBUSMessageHandler.initReceive_NoArgs(msg);

    am_Error_e returnCode = mpIAmCommandReceive->getListMainSources(listSources);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListMainSources called, size is %u", listSources.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listSources);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListMainSinkSoundProperties(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sinkID_t sinkID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    std::vector<am_MainSoundProperty_s> listSinkSoundProperties;
    am_Error_e returnCode = E_NON_EXISTENT;

   ETG_TRACE_USR4(("sinkID:%u",sinkID));
  if(sinkID)
  {
    returnCode = mpIAmCommandReceive->getListMainSinkSoundProperties(sinkID, listSinkSoundProperties);
  }

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListMainSinkSoundProperties called, size is %u", listSinkSoundProperties.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listSinkSoundProperties);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListMainSourceSoundProperties(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sourceID_t sourceID = (am_sourceID_t) mDBUSMessageHandler.getUInt();
    std::vector<am_MainSoundProperty_s> listSinkSoundProperties;
    am_Error_e returnCode = E_NON_EXISTENT;
   ETG_TRACE_USR4(("sourceID:%u",sourceID));
  if(sourceID)
  {
    returnCode = mpIAmCommandReceive->getListMainSourceSoundProperties(sourceID, listSinkSoundProperties);
  }

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListMainSourceSoundProperties called, size is %u", listSinkSoundProperties.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listSinkSoundProperties);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListSourceClasses(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive_NoArgs(msg);
    std::vector<am_SourceClass_s> listSourceClasses;
    am_Error_e returnCode = mpIAmCommandReceive->getListSourceClasses(listSourceClasses);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListSourceClasses called, size is %u", listSourceClasses.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listSourceClasses);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListSinkClasses(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive_NoArgs(msg);

    std::vector<am_SinkClass_s> listSinkClasses;
    am_Error_e returnCode = mpIAmCommandReceive->getListSinkClasses(listSinkClasses);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListSinkClasses called, size is %u", listSinkClasses.size()));

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listSinkClasses);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getListSystemProperties(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive_NoArgs(msg);

    std::vector<am_SystemProperty_s> listSystemProperties;
    am_Error_e returnCode = mpIAmCommandReceive->getListSystemProperties(listSystemProperties);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListSystemProperties called, size is %u", listSystemProperties.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listSystemProperties);
    mDBUSMessageHandler.sendMessage();
}

void IAmCommandReceiverShadow::getTimingInformation(DBusConnection* /* conn */, DBusMessage *msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_mainConnectionID_t mainConnectionID = (am_mainConnectionID_t) mDBUSMessageHandler.getUInt();
    am_timeSync_t delay = 0;
    am_Error_e returnCode = mpIAmCommandReceive->getTimingInformation(mainConnectionID, delay);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getTimingInformation called, mainConnectionID:%u, delay = %d", mainConnectionID, delay))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append((dbus_int16_t) delay);
    mDBUSMessageHandler.sendMessage();
}

DBusHandlerResult IAmCommandReceiverShadow::receiveCallback(DBusConnection *conn, DBusMessage *msg, void *user_data)
{
    ETG_TRACE_USR4(("IAmCommandReceiverShadow::receiveCallback called"))

  assert(conn!=NULL);
    assert(msg!=NULL);
    assert(user_data!=NULL);
    IAmCommandReceiverShadow* reference = (IAmCommandReceiverShadow*) user_data;
    return (reference->receiveCallbackDelegate(conn, msg));
}

void IAmCommandReceiverShadow::sendIntrospection(DBusConnection *conn, DBusMessage *msg)
{
    ETG_TRACE_USR4(("IAmCommandReceiverShadow::sendIntrospection called"));
  assert(conn!=NULL);
    assert(msg!=NULL);
    DBusMessage* reply;
    DBusMessageIter args;
    dbus_uint32_t serial = 0;

    std::ifstream in("CommandInterface.xml", std::ifstream::in);
    if (!in)
    {
    ETG_TRACE_ERR(("IAmCommandReceiverShadow::sendIntrospection could not load xml file"));
        //throw std::runtime_error("IAmCommandReceiverShadow::sendIntrospection Could not load introspecton XML");//Disable reset
    reply  = dbus_message_new_error(msg, DBUS_ERROR_NOT_SUPPORTED , "Introspection file not loaded in file system ");
    }
    else
    {
          // create a reply from the message
          reply = dbus_message_new_method_return(msg);

          std::string introspect((std::istreambuf_iterator<char>(in)), std::istreambuf_iterator<char>());
          const char* string = introspect.c_str();

          // add the arguments to the reply
          dbus_message_iter_init_append(reply, &args);
          if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING, &string))
          {
        ETG_TRACE_ERR(("IAmCommandReceiverShadow::sendIntrospection DBUS handler Out Of Memory!"));
          }
  }
    // send the reply && flush the connection
    if (!dbus_connection_send(conn, reply, &serial))
    {
    ETG_TRACE_ERR(("IAmCommandReceiverShadow::sendIntrospection DBUS handler Out Of Memory!"));
    }
    dbus_connection_flush(conn);

    // free the reply
    dbus_message_unref(reply);
}

DBusHandlerResult IAmCommandReceiverShadow::receiveCallbackDelegate(DBusConnection *conn, DBusMessage *msg)
{
  ETG_TRACE_USR4(("IAmCommandReceiverShadow::receiveCallbackDelegate called"));

    if (dbus_message_is_method_call(msg, DBUS_INTERFACE_INTROSPECTABLE, "Introspect"))
    {
        sendIntrospection(conn, msg);
        return (DBUS_HANDLER_RESULT_HANDLED);
    }

    functionMap_t::iterator iter = mFunctionMap.begin();
    std::string k(dbus_message_get_member(msg));
    iter = mFunctionMap.find(k);
    if (iter != mFunctionMap.end())
    {
        std::string p(iter->first);
      ETG_TRACE_USR4(("method %s need to called",p.c_str()));
        CallBackMethod cb = iter->second;
        (this->*cb)(conn, msg);
        return (DBUS_HANDLER_RESULT_HANDLED);
    }

    return (DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
}

void IAmCommandReceiverShadow::setCommandReceiver(IAmCommandReceive*& receiver)
{
    ETG_TRACE_USR4(("IAmCommandReceiverShadow::setCommandReceiver called"));
  assert(receiver!=NULL);
    mpIAmCommandReceive = receiver;

    gObjectPathVTable.message_function = IAmCommandReceiverShadow::receiveCallback;

    DBusConnection* connection;
    mpIAmCommandReceive->getDBusConnectionWrapper(mpCAmDbusWrapper);
    assert(mpCAmDbusWrapper!=NULL);

    mpCAmDbusWrapper->getDBusConnection(connection);
    assert(connection!=NULL);
    mDBUSMessageHandler.setDBusConnection(connection);

    std::string path(MY_NODE);
    mpCAmDbusWrapper->registerCallback(&gObjectPathVTable, path, this);

}

void am::IAmCommandReceiverShadow::getListSinkMainNotificationConfigurations(DBusConnection* /* conn */, DBusMessage* msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sinkID_t sinkID = static_cast<am_sinkID_t>(mDBUSMessageHandler.getUInt());
    std::vector<am_NotificationConfiguration_s> listNotificationConfigurations;
    am_Error_e returnCode = mpIAmCommandReceive->getListMainSinkNotificationConfigurations(sinkID,listNotificationConfigurations);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListSinkMainNotificationConfigurations called, sinkID:%u, size is %u"
          , sinkID, listNotificationConfigurations.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listNotificationConfigurations);
    mDBUSMessageHandler.sendMessage();
}

void am::IAmCommandReceiverShadow::getListSourceMainNotificationConfigurations(DBusConnection* /* conn */, DBusMessage* msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sourceID_t sourceID = static_cast<am_sourceID_t>(mDBUSMessageHandler.getUInt());
    std::vector<am_NotificationConfiguration_s> listNotificationConfigurations;
    am_Error_e returnCode = mpIAmCommandReceive->getListMainSourceNotificationConfigurations(sourceID, listNotificationConfigurations);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::getListSourceMainNotificationConfigurations called, sourceID:%u, size is %u"
          , sourceID, listNotificationConfigurations.size()))

    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.append(listNotificationConfigurations);
    mDBUSMessageHandler.sendMessage();

}

void am::IAmCommandReceiverShadow::setSinkMainNotificationConfiguration(DBusConnection* /* conn */, DBusMessage* msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sinkID_t sinkID = (am_sinkID_t) mDBUSMessageHandler.getUInt();
    dbus_int16_t type = 0;
    dbus_int16_t status = 0;
    dbus_int16_t parameter = 0;
    mDBUSMessageHandler.getNotificationConfiguration(type, status, parameter);
    am_NotificationConfiguration_s mainNotificationConfiguration;
    mainNotificationConfiguration.type = static_cast<am_CustomNotificationType_t> (type);
    mainNotificationConfiguration.status = static_cast<am_NotificationStatus_e> (status);
    mainNotificationConfiguration.parameter = static_cast<int16_t>(parameter);

   ETG_TRACE_USR2(("IAmCommandReceiverShadow::setSinkMainNotificationConfiguration called, sinkID:%u type:%u status:%u parameter:%d"
         ,sinkID, (uint16_t)mainNotificationConfiguration.type, (uint16_t)mainNotificationConfiguration.status, mainNotificationConfiguration.parameter))

    am_Error_e returnCode = mpIAmCommandReceive->setMainSinkNotificationConfiguration(sinkID,mainNotificationConfiguration);
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}

void am::IAmCommandReceiverShadow::setSourceMainNotificationConfiguration(DBusConnection* /* conn */, DBusMessage* msg)
{
    vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

    mDBUSMessageHandler.initReceive(msg);
    am_sourceID_t sourceID = (am_sourceID_t) mDBUSMessageHandler.getUInt();
    dbus_int16_t type = 0;
    dbus_int16_t status = 0;
    dbus_int16_t parameter = 0;
    mDBUSMessageHandler.getNotificationConfiguration(type, status, parameter);
    am_NotificationConfiguration_s mainNotificationConfiguration;
    mainNotificationConfiguration.type = static_cast<am_CustomNotificationType_t> (type);
    mainNotificationConfiguration.status = static_cast<am_NotificationStatus_e> (status);
    mainNotificationConfiguration.parameter = static_cast<int16_t>(parameter);

    ETG_TRACE_USR2(("IAmCommandReceiverShadow::setSourceMainNotificationConfiguration called, sourceID:%d type:%u status:%u parameter:%d"
         , sourceID, (uint16_t)mainNotificationConfiguration.type, (uint16_t)mainNotificationConfiguration.status, mainNotificationConfiguration.parameter))


    am_Error_e returnCode = mpIAmCommandReceive->setMainSourceNotificationConfiguration(sourceID,mainNotificationConfiguration);
    mDBUSMessageHandler.initReply(msg);
    mDBUSMessageHandler.append((dbus_int16_t) returnCode);
    mDBUSMessageHandler.sendMessage();
}
IAmCommandReceiverShadow::functionMap_t IAmCommandReceiverShadow::createMap()
{
    ETG_TRACE_USR1(("IAmCommandReceiverShadow::createMap called"));

  functionMap_t m;
    m["Connect"] = &IAmCommandReceiverShadow::connect;
    m["Disconnect"] = &IAmCommandReceiverShadow::disconnect;
    m["SetVolume"] = &IAmCommandReceiverShadow::setVolume;
    m["VolumeStep"] = &IAmCommandReceiverShadow::volumeStep;
    m["SetSinkMuteState"] = &IAmCommandReceiverShadow::setSinkMuteState;
    m["SetMainSinkSoundProperty"] = &IAmCommandReceiverShadow::setMainSinkSoundProperty;
    m["SetMainSourceSoundProperty"] = &IAmCommandReceiverShadow::setMainSourceSoundProperty;
    m["GetListMainConnections"] = &IAmCommandReceiverShadow::getListMainConnections;
    m["GetListMainSinks"] = &IAmCommandReceiverShadow::getListMainSinks;
    m["GetListMainSources"] = &IAmCommandReceiverShadow::getListMainSources;
    m["GetListMainSinkSoundProperties"] = &IAmCommandReceiverShadow::getListMainSinkSoundProperties;
    m["GetListMainSourceSoundProperties"] = &IAmCommandReceiverShadow::getListMainSourceSoundProperties;
    m["GetListSourceClasses"] = &IAmCommandReceiverShadow::getListSourceClasses;
    m["GetListSinkClasses"] = &IAmCommandReceiverShadow::getListSinkClasses;
    m["GetListSystemProperties"] = &IAmCommandReceiverShadow::getListSystemProperties;
    m["GetTimingInformation"] = &IAmCommandReceiverShadow::getTimingInformation;
    m["SetSystemProperty"] = &IAmCommandReceiverShadow::setSystemProperty;
    m["getListSinkMainNotificationConfigurations"] = &IAmCommandReceiverShadow::getListSinkMainNotificationConfigurations;
    m["getListSourceMainNotificationConfigurations"] = &IAmCommandReceiverShadow::getListSourceMainNotificationConfigurations;
    m["setSinkMainNotificationConfiguration"] = &IAmCommandReceiverShadow::setSinkMainNotificationConfiguration;
    m["setSourceMainNotificationConfiguration"] = & IAmCommandReceiverShadow::setSourceMainNotificationConfiguration;
    return (m);
}

